/*____________________________________________________________________________
        Copyright (C) 2000 Networks Associates Technology, Inc.
        All rights reserved.

		Check and create keyring signatures
		
        $Id: pgpKeySig.c,v 1.34.2.2 2001/05/29 17:47:30 hal Exp $
____________________________________________________________________________*/
#include "pgpConfig.h"


#include "pgpDebug.h"
#include "pgpMakeSig.h"
#include "pgpMem.h"
#include "pgpPktByte.h"
#include "pgpRegExp.h"
#include "pgpKeyPriv.h"
#include "pgpTimeDate.h"
#include "pgpHashPriv.h"
#include "pgpErrors.h"
#include "pgpPubKey.h"
#include "pgpSig.h"
#include "pgpSigSpec.h"
#include "pgpTrustPriv.h"
#include "pgpUsuals.h"
#include "pgpTypes.h"
#include "pgpX509Priv.h"
#include "pgpOptionListPriv.h"


/*
 * Keyring signature checking code.
 */



/* Callback for checking function */
typedef struct PGPKeyCheckState {
	PGPContextRef			context;
	PGPEventHandlerProcPtr	progress;
	PGPUserValue			userValue;
	PGPUInt32				total;
	PGPUInt32				sofar;
} PGPKeyCheckState;



static PGPError
sCheckCRLs( PGPKeySet *keyset, int allflag, PGPKeyDBRef otherdb,
	PGPKeySetRef changedSet );

static PGPBoolean
sCheckableSig( PGPKeyDBObj const *sig, PGPKeyDB *otherdb,
	PGPBoolean allflag, PGPBoolean revocationonly, PGPKeyDBObj **retsignkey);

	static void
sUncheckSig( PGPKeyDBObj *sig, PGPKeyDB *keydb, PGPBoolean *changed );

	static PGPError
sCheckSig(PGPKeyDBObj *sig, PGPKeyDBObj *key, PGPBoolean *changed,
	PGPBoolean *verified );





	static PGPError
sCheckKeysCallback(void *arg, PGPKeyDBObj *obj, int status)
{
	PGPKeyCheckState		*s = (PGPKeyCheckState *)arg;
	PGPError				err = kPGPError_NoErr;
	PGPOptionListRef		newOptionList = NULL;

	(void) obj;
	(void) status;

	if (IsntNull (s->progress)) {
		err = pgpEventNull (s->context, &newOptionList,
							s->progress, s->userValue, ++s->sofar, s->total);
		if (IsntNull (newOptionList))
			pgpFreeOptionList (newOptionList);
	}
	return err;
}
	

/* Called on the back end from the front end preparatory to doing a one
 * key at a time series of key checks.
 */
	PGPError
pgpPrepareToCheckKeyRingSigs_internal( PGPKeySetRef keysToCheck,
	PGPKeyDBRef otherDB, PGPBoolean checkAll, PGPUInt32 *nsigs,
	PGPKeySetRef changedSet )
{
	PGPError err;

	*nsigs = pgpCountCheckableKeySigs( keysToCheck, otherDB, checkAll, FALSE );
	err = sCheckCRLs( keysToCheck, checkAll, NULL, changedSet );
	if( IsPGPError( err ) )
		return err;
	if( IsntNull( otherDB ) )
	{
		/* Must check CRLs in otherdb against this one */
		PGPKeySetRef oroot = PGPPeekKeyDBRootKeySet( otherDB );
		PGPKeyDBRef keydb = PGPPeekKeySetKeyDB( keysToCheck );
		if( IsPGPError( err = sCheckCRLs( oroot, checkAll,
										  keydb, changedSet ) ) )
			return err;
	}
	return kPGPError_NoErr;
}


	PGPError
pgpCheckKeyRingSigs_internal( PGPKeySetRef keysToCheck, PGPKeyDBRef otherDB,
	PGPBoolean checkAll, PGPEventHandlerProcPtr eventHandler,
	PGPUserValue eventHandlerData, PGPKeySetRef changeSet )
{
	PGPKeyCheckState		s;

	s.context = PGPPeekKeySetContext( keysToCheck );
	s.progress = eventHandler;
	s.userValue = eventHandlerData;
	s.sofar = 0;
	if( IsntNull( eventHandler ) )
	{
		s.total = pgpCountCheckableKeySigs( keysToCheck, otherDB, checkAll,
											FALSE );
	}

	return pgpCheckKeySigs( keysToCheck, otherDB, checkAll, FALSE, FALSE,
							sCheckKeysCallback, &s, changeSet );
}

	PGPError
PGPCheckKeyRingSigs( PGPKeySetRef keysToCheck, PGPKeyDBRef otherDB,
	PGPBoolean checkAll, PGPEventHandlerProcPtr eventHandler,
	PGPUserValue eventHandlerData )
{
	PGPKeyDBRef				setDB;
	PGPKeySetRef			changedSet;
	PGPError				err;
	PGPUInt32				nsigs;
	PGPKeyCheckState		s;

	PGPValidateKeySet( keysToCheck );
	if( IsntNull( otherDB ) )
		PGPValidateKeyDB( otherDB );

	pgpEnterPGPErrorFunction();

	setDB = PGPPeekKeySetKeyDB( keysToCheck );
	if( otherDB == setDB )
		otherDB = NULL;
	
	if( pgpFrontEndKeyDB( setDB ) )
	{
		PGPUInt32 *keylist;
		PGPSize keylistsize;
		PGPUInt32 *changelist;
		PGPSize changelistsize;
		PGPContextRef context = PGPPeekKeyDBContext( setDB );

		if( IsNull( eventHandler ) )
		{
			/* Do it all in background if he doesn't want events */

			err = pgpKeySetFlatten( keysToCheck, &keylist, &keylistsize );
			if( IsPGPError( err ) )
				return err;
			err = pgpCheckKeyRingSigs_back( context, setDB->id, keylist,
											keylistsize,
											(IsNull(otherDB)?0:otherDB->id),
											checkAll, &changelist,
											&changelistsize );
			if( IsPGPError( err ) )
				return err;
			err = pgpKeyRefreshFromList( setDB, changelist, changelistsize );
			if( IsPGPError( err ) )
				return err;
			return PGPCalculateTrust( keysToCheck, otherDB );
		}
		/* Else he wants events, must do it one at a time */
		err = pgpKeySetFlatten( keysToCheck, &keylist, &keylistsize );
		if( IsPGPError( err ) )
			return err;
		err = pgpPrepareToCheckKeyRingSigs_back( context, setDB->id,
											keylist, keylistsize,
											(IsNull(otherDB)?0:otherDB->id),
											checkAll, &nsigs, &changelist,
											&changelistsize );
		if( IsPGPError( err ) )
			return err;
		if( IsPGPError( err = pgpKeySetUnflattenFree( setDB, changelist,
													  changelistsize,
													  &changedSet ) ) )
			return err;
						
		
		s.context = context;
		s.progress = eventHandler;
		s.userValue = eventHandlerData;
		s.sofar = 0;
		s.total = nsigs;

		err = pgpCheckKeySigs( keysToCheck, otherDB, checkAll, FALSE, TRUE,
							   sCheckKeysCallback, &s, changedSet );
		if( IsPGPError( err ) )
		{
			PGPFreeKeySet( changedSet );
			return err;
		}
		if( IsPGPError( err = pgpKeySetRefreshFree( changedSet ) ) )
			return err;
		return PGPCalculateTrust( keysToCheck, otherDB );
	}

	err = pgpCheckKeyRingSigs_internal( keysToCheck, otherDB, checkAll,
										 eventHandler, eventHandlerData,
										 NULL );
	if( IsPGPError( err ) )
		return err;

	return PGPCalculateTrust( keysToCheck, otherDB );
}




/*
 * Count the number of sigs we would check with pgpCheckKeySigs.
 * The return value is the number of times the progress func() function
 * will be called by pgpCheckKeySigs().
 * NOTE: this tries to be as accurate as possible, but user code
 * should not die if it is slightly incorrect.
 */
	PGPUInt32
pgpCountCheckableKeySigs(PGPKeySet *keyset, PGPKeyDB *otherdb,
	PGPBoolean allflag, PGPBoolean revocationonly)
{
	PGPKeyDB *keydb;
	PGPKeyDBObj  *key;
	PGPKeyDBObj  *child;
	PGPKeyDBObj  *gchild;
	PGPKeyDBObj *signkey;
	PGPUInt32 count = 0;

	keydb = PGPPeekKeySetKeyDB( keyset );

	for( key = keydb->firstKeyInDB; IsntNull( key ); key = key->next)
	{
		if( !pgpKeyDBObjIsReal( key ) )
			continue;
		if( !pgpKeySetIsMember( key, keyset ) )
			continue;
		for( child = key->down; IsntNull( child ); child = child->next)
		{
			if( !pgpKeyDBObjIsReal( child ) )
				continue;
			if( !pgpKeySetIsMember( child, keyset ) )
				continue;
			if( pgpObjectType( child ) == RINGTYPE_SIG &&
				sCheckableSig( child, otherdb,
							   allflag, revocationonly, &signkey ) )
			{
				if( IsntNull( signkey ) )
					++count;
			}
			for( gchild = child->down; IsntNull(gchild); gchild = gchild->next)
			{
				if( !pgpKeyDBObjIsReal( gchild ) )
					continue;
				if( !pgpKeySetIsMember( gchild, keyset ) )
					continue;
				if( pgpObjectType( gchild ) == RINGTYPE_SIG &&
					sCheckableSig( gchild, otherdb,
								   allflag, revocationonly, &signkey ) )
				{
					if( IsntNull( signkey ) )
						++count;
				}
			}
		}
	}
	return count;
}


/* True if signature would be checked given the specified parameters */
/* Return *retsignkey as signing key if we return TRUE */
static PGPBoolean
sCheckableSig( PGPKeyDBObj const *sig, PGPKeyDB *otherdb,
	PGPBoolean allflag, PGPBoolean revocationonly, PGPKeyDBObj **retsignkey)
{
	PGPKeyDBObj *		signkey;
	PGPKeyID 			signkeyid;
	PGPSigInfo *		sinfo;
	PGPByte				sigalg;
	PGPError			err;

	if( IsntNull( retsignkey ) )
		*retsignkey = NULL;

	pgpAssert( pgpObjectType( sig ) == RINGTYPE_SIG );

	sinfo = pgpSigToSigInfo( sig );
	signkey = sinfo->by;
	if( !pgpKeyDBObjIsReal( signkey ) )
	{
		if( IsNull( otherdb ) )
		{
			/* leave *retsignkey as NULL so caller knows we don't have it */
			return TRUE;
		}
		/* Else see if it is in other db */
		pgpSigID8( sig, &sigalg, &signkeyid );
		err = pgpGetKeyByKeyID(otherdb, &signkeyid, TRUE, FALSE, &signkey );
		if( IsPGPError( err ) )
			return FALSE;
		if( !pgpKeyDBObjIsReal( signkey ) )
		{
			/* leave *retsignkey as NULL so caller knows we don't have it */
			return TRUE;
		}

	}
	if ( (allflag || !(sinfo->trust & (PGP_SIGTRUSTF_CHECKED_TRIED |
									   PGP_SIGTRUSTF_REVOKEDBYCRL)))
		 && (!revocationonly || sinfo->type == PGP_SIGTYPE_KEY_REVOKE) )
	{
		if( IsntNull( retsignkey ) )
			*retsignkey = signkey;
		return TRUE;
	}
	
	return FALSE;
}



/* Compare an object with a buffer */
	static PGPBoolean
sObjDiffers( PGPKeyDBObj *obj, PGPByte *buf, PGPSize len )
{
	PGPBoolean mustFree = FALSE;
	PGPBoolean match = FALSE;
	PGPByte const *objdata;
	PGPSize objlen;
	PGPSize hlen;

	/* Fetch public version of secret key for this */
	if( OBJISKEY( obj ) && pgpKeyIsSec( obj ) )
	{
		objdata = pgpKeyDBObjToPubData( obj, &objlen );
		mustFree = TRUE;
	} else {
		objdata = pgpFetchObject( obj, &objlen );
	}

	hlen = pgpPktBufferHeaderLen( objdata );

	if( len == objlen-hlen && pgpMemoryEqual( objdata+hlen, buf, len ) )
		match = TRUE;

	if( mustFree )
		PGPFreeData( (PGPByte *)objdata );

	return !match;
}


/* True if X.509 sig is valid */
	static PGPBoolean
pgpSigX509Valid (PGPKeyDBObj *obj, PGPKeyDBObjRef key )
{
	PGPASN_Certificate *cert = NULL;
	PGPPubKey const *pub = NULL;
	PGPByte *buf;
	PGPByte *x509cert;
	PGPSize len;
	PGPSize x509len;
	PGPKeyDBRef keydb;
	PGPContextRef context;
	PGPKeyDBObj *parentkey;
	PGPKeyDBObj *parentuserid;
	PGPByte loversion;
	MemPool mempoolinit;
	PGPError err = kPGPError_NoErr;

	/*
	 * Steps:
	 * 1) Compare the key and userid data with that in the cert.
	 * 2) Validate the signature in the cert, using pub
	 */

	keydb = PGPPeekKeyDBObjKeyDB( obj );
	context = PGPPeekKeyDBContext( keydb );
	mempoolinit = keydb->objPool;

	pub = pgpKeyPubKey( key, PGP_PKUSE_SIGN );
	if( IsNull( pub ) )
		goto error;

	parentuserid = obj->up;
	if( !OBJISUSERID(parentuserid) )
		goto error;
	parentkey = parentuserid->up;
	if( !OBJISTOPKEY(parentkey) )
		goto error;

	/* Find embedded X509 cert */
	buf = (PGPByte *)pgpFetchObject(obj, &len);
	if( IsNull( buf )  )
		goto error;
	x509cert = (PGPByte *)pgpSigFindNAISubSubpacket(buf, SIGSUBSUB_X509, 0,
										&x509len, NULL, NULL, NULL, NULL);
	if( IsNull( x509cert )  )
		goto error;

	/* Check type, version bytes */
	if( x509len < 3 )
		goto error;
	pgpAssert (x509cert[0] == SIGSUBSUB_X509);
	if( x509cert[1] != SIGSUBSUB_X509_VERSION_HI )
		goto error;
	loversion = x509cert[2];
	x509cert += 3;
	x509len -= 3;

	err = pgpX509VerifySignedObject( context, pub, x509cert, x509len );
	if( IsPGPError(err ) )
		goto error;
	pgpPubKeyDestroy( (PGPPubKey *)pub );
	pub = NULL;

	/* Process cert into a structure using ASN.1 library */
	err = pgpX509BufferToCert( context, x509cert, x509len, &cert );
	if( IsPGPError( err ) )
		goto error;

	/* At this point we can destroy the raw X.509 buffer data */
	x509cert = NULL;

	/* Check to see if data from X.509 cert matches parent key and userid */

	err = pgpX509CertToKeyBuffer(context, &keydb->objPool, keydb, cert,
								 loversion, &buf, &len);
	if( IsPGPError( err )  ||  sObjDiffers( parentkey, buf, len ) )
		goto error;

	err = pgpX509CertToUserIDBuffer (context, &keydb->objPool, cert, loversion,
									 &buf, &len );
	if( IsPGPError( err )  ||  sObjDiffers( parentuserid, buf, len ) )
		goto error;

	err = pgpX509CertToSigBuffer (context, &keydb->objPool, cert, loversion,
								  &buf, &len );
	if( IsPGPError( err )  ||  sObjDiffers( obj, buf, len ) )
		goto error;

	pgpX509FreeCert( context, cert );
	memPoolCutBack( &keydb->objPool, &mempoolinit );

	/* All OK */
	return TRUE;

 error:
	if( IsntNull( pub ) )
		pgpPubKeyDestroy( (PGPPubKey *)pub );
	if( IsntNull( cert ) )
		pgpX509FreeCert( context, cert );
	memPoolCutBack( &keydb->objPool, &mempoolinit );
	return FALSE;
}



/*
 * Mark all the signatures in its keydb that crl revokes as REVOKEDBYCRL.
 * crl's parent is "key".
 * If otherdb is not null, ONLY mark the ones there.  In that case
 * return changedSet as being the changed keys in that set as it is the
 * primary one of interest.  (The idea here is that this is used for the
 * case where "otherdb" is really the main db and the key is a signer
 * in another keydb.)
 */
	static PGPError
sPropagateOneCRL( PGPKeyDBObj *key, PGPKeyDBObj *crl, PGPKeyDBRef otherdb,
	PGPKeySetRef changedSet )
{
	PGPASN_Certificate *cert = NULL;
	PGPASN_CertificateRevocationList *crl509 = NULL;
	PGPASN_revokedCertificates_SEQ_OF *revs;
	PGPASN_INTEGER *certnum, *revnum;
	PGPByte *buf;
	PGPSize len;
	PGPByte *x509cert;
	PGPSize x509len;
	PGPKeyDBObj *sig;
	PGPInt32 i;
	PGPInt32 count;
	PGPInt32 numrevsleft;
	PGPKeyInfo *kinfo;
	PGPKeyDBObj *pairkey = NULL;
	PGPSigInfo *sinfo = NULL;
	PGPKeyDB *keydb;
	PGPContextRef context;
	PGPError err = kPGPError_NoErr;
	
	pgpAssert (OBJISKEY(key));
	pgpAssert (OBJISCRL(crl));

	keydb = PGPPeekKeyDBObjKeyDB( crl );
	context = PGPPeekKeyDBContext( keydb );
	kinfo = pgpKeyToKeyInfo( key );

	buf = (PGPByte *)pgpFetchObject( crl, &len );
	if( IsNull(buf) )
	{
		err = pgpKeyDBError( keydb );
		goto error;
	}

	buf = (PGPByte *)pgpCRLFindCRL( buf, len, &len );
	if( IsNull( buf ) )
	{
		/* Unknown CRL type, ignore it */
		err = kPGPError_NoErr;
		goto error;
	}
	err = pgpX509BufferToCRL( context, buf, len, &crl509 );
	if( IsPGPError( err ) )
		goto error;

	if( IsNull( crl509->tbsCertList.revokedCertificates ) )
	{
		/* Nothing to do */
		goto error;
	}

	revs = crl509->tbsCertList.revokedCertificates;
	count = revs->n;
	if( count == 0 )
	{
		/* Nothing to do */
		goto error;
	}
	numrevsleft = count;

	/* Find corresponding key in otherdb if it exists */
	if( IsntNull( otherdb ) )
	{
		pairkey = pgpFindPairedX509Key( key, otherdb );
		/* If no such key in otherdb, we are done */
		if( IsNull( pairkey ) )
			goto error;
		kinfo = pgpKeyToKeyInfo( pairkey );
		/* By setting kinfo we will look in otherdb for sigs to propagate */
	}
		
	/* Check all sigs issued by key to see if any are revoked */
	for( sig=kinfo->sigsby; IsntNull(sig); sig=sinfo->nextby )
	{
		pgpAssert( OBJISSIG( sig ) );
		sinfo = pgpSigToSigInfo(sig);
		if( !SIGISX509(sinfo) )
			continue;
		buf = (PGPByte *)pgpFetchObject( sig, &len );
		if( IsNull(buf) )
		{
			err = pgpKeyDBError( keydb );
			goto error;
		}

		x509cert = (PGPByte *)pgpSigFindNAISubSubpacket(buf,
								SIGSUBSUB_X509, 0, &x509len, NULL, NULL,
								NULL, NULL);
		if( IsNull( x509cert )  )
			continue;

		/* Skip type, version bytes */
		pgpAssert (x509cert[0] == SIGSUBSUB_X509);
		x509cert += 3;
		x509len -= 3;

		/* Process cert into a structure using ASN.1 library */
		err = pgpX509BufferToCert( context, x509cert, x509len, &cert );
		if( IsPGPError( err ) )
			continue;

		/* Get pointer to cert's serial number, which we will compare */
		certnum = &cert->tbsCertificate.serialNumber;

		/* Compare against each entry in the CRL */
		for (i=0; i<count; ++i)
		{
			revnum = &revs->elt[i]->userCertificate;
			if( certnum->len == revnum->len &&
					pgpMemoryEqual( certnum->val, revnum->val, revnum->len) )
			{
				/* Have a match, cert is revoked */
				if( IsntNull( changedSet ) &&
							!(sinfo->trust & PGP_SIGTRUSTF_REVOKEDBYCRL) )
					PGPAddKey( PGPPeekKeyDBObjKey(sig), changedSet );
				sinfo->trust |= PGP_SIGTRUSTF_REVOKEDBYCRL;
				if( --numrevsleft == 0 )
				{
					/* We have accounted for all revocations */
					err = kPGPError_NoErr;
					goto error;
				}
				break;
			}
		}
		pgpX509FreeCert( context, cert );
		cert = NULL;
	}

error:
	if( IsntNull( cert ) )
		pgpX509FreeCert( context, cert );
	if( IsntNull( crl509 ) )
		pgpX509FreeCRL( context, crl509 );

	return err;
}




/*
 * Check one CRL on a keyring.  Set its trust flags to show whether the
 * signature checks OK.
 */
	static PGPError
sCheckOneCRL( PGPKeyDBObj *key, PGPKeyDBObj *crl, PGPKeySetRef changedSet )
{
	PGPByte *buf;
	PGPSize len;
	PGPPubKey *pub = NULL;
	PGPKeyDB *keydb;
	PGPContextRef context;
	PGPCRLInfo *cinfo;
	PGPError err = kPGPError_NoErr;

	pgpAssert (OBJISKEY(key));
	pgpAssert (OBJISCRL(crl));

	keydb = PGPPeekKeyDBObjKeyDB( key );
	context = PGPPeekKeyDBContext( keydb );
	cinfo = pgpCRLToCRLInfo( crl );

	/* Get the pubkey struct for signing key */
	pub = pgpKeyPubKey(key, PGP_PKUSE_SIGN);
	if( IsNull(pub) )
	{
		err = pgpKeyDBError(keydb);
		if( pgpIsKeyRelatedError( err ) )
			err = kPGPError_NoErr;
		goto error;
	}

	buf = (PGPByte *)pgpFetchObject( crl, &len );
	if( IsNull(buf) )
	{
		err = pgpKeyDBError(keydb);
		goto error;
	}

	buf = (PGPByte *)pgpCRLFindCRL( buf, len, &len );
	err = pgpX509VerifySignedObject( context, pub, buf, len );

	if( IsPGPError(err ) )
	{
		if( err == kPGPError_X509InvalidCertificateSignature  )
		{
			if( IsntNull( changedSet ) &&
				(cinfo->trust & PGP_SIGTRUSTF_CHECKED_TRIED) !=
													PGP_SIGTRUSTF_TRIED )
				PGPAddKey( key, changedSet );
			cinfo->trust &= ~PGP_SIGTRUSTF_CHECKED;
			cinfo->trust |= PGP_SIGTRUSTF_TRIED;
		}
		err = kPGPError_NoErr;
	} else {
		if( IsntNull( changedSet ) &&
			(cinfo->trust & PGP_SIGTRUSTF_CHECKED_TRIED) !=
											PGP_SIGTRUSTF_CHECKED_TRIED )
			PGPAddKey( key, changedSet );
		cinfo->trust |= PGP_SIGTRUSTF_CHECKED_TRIED;
	}
	
error:	
	if( IsntNull( pub ) )
		pgpPubKeyDestroy( pub );
	return err;
}


/*
 * Check the CRLs on a keyring.  If allflag is true, recheck all, else
 * only check unchecked ones.  Propagate revocations to sigs in keyset.
 * If otherdb is not null, we return the changedSet as keys in that
 * set, as it is the primary one of interest.
 */
	static PGPError
sCheckCRLs( PGPKeySet *keyset, int allflag, PGPKeyDBRef otherdb,
	PGPKeySetRef changedSet )
{
	PGPKeyDBObj *key = NULL;
	PGPKeyDB *keydb;
	PGPKeyDBObj *obj;
	PGPCRLInfo *cinfo;
	PGPError err = kPGPError_NoErr;

	keydb = PGPPeekKeySetKeyDB( keyset );

	for( key = keydb->firstKeyInDB; IsntNull( key ); key = key->next )
	{
		if( !pgpKeyDBObjIsReal( key ) )
			continue;
		if( !pgpKeySetIsMember( key, keyset ) )
			continue;
		for( obj = key->down; IsntNull( obj ); obj = obj->next )
		{
			if( !pgpKeyDBObjIsReal( obj ) )
				continue;
			if( !OBJISCRL(obj) )
				continue;

			cinfo = pgpCRLToCRLInfo( obj );

			if( allflag || !(cinfo->trust & PGP_SIGTRUSTF_CHECKED_TRIED) )
			{
				pgpAssert( IsntNull( key ) );
				err = sCheckOneCRL (key, obj,
									IsntNull(otherdb)?changedSet:NULL);
				if( IsPGPError( err ) )
					goto error;
			}
			if( cinfo->trust & PGP_SIGTRUSTF_CHECKED )
			{
				err = sPropagateOneCRL (key, obj, otherdb, changedSet);
				if( IsPGPError( err ) )
					goto error;
			}
		}
	}
error:
	return err;
}



/*
 * Check the signatures in a keydb.  Checks only unchecked signatures,
 * or all signatures if allflag is true.  If revocationonly is set,
 * only revocation signatures are checked (allflag again controls whether
 * previously checked rev sigs are rechecked).
 *
 * Signatures may be made either by keys in keydb or in otherdb.
 *
 * If func is non-null, calls it with each checked signature, after
 * checking.  This is for progress indications.  If func returns a
 * negative value, the check is aborted.
 *
 * If frontEnd is true we are called on the front end and must do it all
 * on the back end.
 */
	PGPError
pgpCheckKeySigs(PGPKeySet *keyset, PGPKeyDB *otherdb,
	PGPBoolean allflag, PGPBoolean revocationonly, PGPBoolean frontEnd,
	PGPError (*func)(void *, PGPKeyDBObj *, int), void *arg,
	PGPKeySetRef changeSet )
{
	PGPContextRef context;
	PGPKeyDB	*keydb;
	PGPKeyDBObj *key;
	PGPKeyDBObj *child;
	PGPKeyDBObj *gchild;
	PGPUInt32	setID;
	PGPUInt32	otherID;
	PGPBoolean	handled, verified, changed;
	PGPError err = kPGPError_NoErr;

	if( frontEnd )
		pgpAssert( IsntNull( func ) );
	keydb = PGPPeekKeySetKeyDB( keyset );
	context = PGPPeekKeyDBContext( keydb );
	setID = keydb->id;
	otherID = IsNull(otherdb) ? 0 : otherdb->id;

	if( !frontEnd )
	{
		if( IsPGPError( err = sCheckCRLs( keyset, allflag, NULL, changeSet )))
			goto error;
		if( IsntNull( otherdb ) )
		{
			/* Must check CRLs in otherdb against this one */
			PGPKeySetRef oroot = PGPPeekKeyDBRootKeySet( otherdb );
			if( IsPGPError( err = sCheckCRLs( oroot, allflag,
											  keydb, changeSet ) ) )
				goto error;
		}
	}

	for( key = keydb->firstKeyInDB; IsntNull( key ); key = key->next)
	{
		if( !pgpKeyDBObjIsReal( key ) )
			continue;
		if( !pgpKeySetIsMember( key, keyset ) )
			continue;
		for( child = key->down; IsntNull( child ); child = child->next)
		{
			if( !pgpKeyDBObjIsReal( child ) )
				continue;
			if( !pgpKeySetIsMember( child, keyset ) )
				continue;
			if( pgpObjectType( child ) == RINGTYPE_SIG )
			{
				if( frontEnd )
				{
					err = pgpCheckSig_back( context, setID,
											pgpKeyDBObjID(child), otherID,
											allflag, revocationonly,
											&handled, &changed, &verified);
				} else {
					err = pgpCheckSig_internal( keydb, child, otherdb,
												allflag, revocationonly,
												&handled, &changed,
												&verified);
				}
				if( IsPGPError( err ) )
					goto error;
				if( changed && IsntNull(changeSet) )
					PGPAddKey( key, changeSet );
				if( handled && IsntNull(func) )
				{
					err = func( arg, child, verified );
					if( IsPGPError( err ) )
						goto error;
				}
			}
			for( gchild = child->down; IsntNull(gchild); gchild = gchild->next)
			{
				if( !pgpKeyDBObjIsReal( gchild ) )
					continue;
				if( !pgpKeySetIsMember( gchild, keyset ) )
					continue;
				if( pgpObjectType( gchild ) == RINGTYPE_SIG )
				{
					if( frontEnd )
					{
						err = pgpCheckSig_back( context, setID,
												pgpKeyDBObjID(gchild), otherID,
												allflag, revocationonly,
												&handled, &changed, &verified);
					} else {
						err = pgpCheckSig_internal( keydb, gchild, otherdb,
													allflag, revocationonly,
													&handled, &changed,
													&verified);
					}
					if( IsPGPError( err ) )
						goto error;
					if( changed && IsntNull(changeSet) )
						PGPAddKey( key, changeSet );
					if( handled && IsntNull(func) )
					{
						err = func( arg, gchild, verified );
						if( IsPGPError( err ) )
							goto error;
					}
				}
			}
		}
	}
 error:
	return err;
}


/* Mark sig as unchecked */
	static void
sUncheckSig( PGPKeyDBObj *sig, PGPKeyDB *keydb, PGPBoolean *changed )
{
	PGPSigInfo *	sinfo;
	PGPByte			strust;

	*changed = FALSE;
	sinfo = pgpSigToSigInfo( sig );
	strust = sinfo->trust;
	strust &= ~PGP_SIGTRUSTF_CHECKED_TRIED;
	if( strust != sinfo->trust )
	{
		sinfo->trust = strust;
		pgpKeyDBIsDirty( keydb );
		*changed = TRUE;
	}
}


	PGPError
pgpCheckSig_internal( PGPKeyDBRef keydb, PGPKeyDBObj *sig,
	PGPKeyDBRef otherdb, PGPBoolean checkAll, PGPBoolean revocationonly,
	PGPBoolean *handled, PGPBoolean *changed, PGPBoolean *verified )
{
	PGPKeyDBObj *	signkey;
	PGPError		err = kPGPError_NoErr;

	*handled = FALSE;
	*changed = FALSE;
	*verified = FALSE;
	if( sCheckableSig( sig, otherdb, checkAll, revocationonly, &signkey ) )
	{
		if( IsNull( signkey ) )
		{
			sUncheckSig( sig, keydb, changed );
		} else {
			*handled = TRUE;
			err = sCheckSig( sig, signkey, changed, verified );
		}
	}
	return err;
}


/* Check sig */
	static PGPError
sCheckSig( PGPKeyDBObj *sig, PGPKeyDBObj *key, PGPBoolean *changed,
		   PGPBoolean *verified )
{
	PGPKeyInfo *	kinfo;
	PGPSigInfo *	sinfo;
	PGPKeyDB *		keydb;
	PGPKeyDB *		sigdb;
	PGPKeyDBObj *	parent;
	PGPKeyDBObj *	topkey;
	PGPKeyDBObj *	parentkey;
	PGPUserIDInfo *	uinfo;
	PGPContextRef	context;
	PGPByte			strust;
	PGPBoolean		sighashnew;
	PGPHashVTBL const *	hash;
	PGPHashContextRef	hashctx = NULL;
	PGPByte const *	buf;
	PGPByte const *	extrabuf;
	PGPSize			len;
	PGPSize			hlen;
	PGPSize			extralen;
	PGPError		err = kPGPError_NoErr;
	MemPool			cut;
    PGPMemoryMgrRef	mgr = NULL;

	pgpAssert( OBJISSIG( sig ) );
	pgpAssert( OBJISKEY( key ) );

	*changed = FALSE;
	*verified = FALSE;
	keydb = PGPPeekKeyDBObjKeyDB( key );
	sigdb = PGPPeekKeyDBObjKeyDB( sig );
	context = PGPPeekKeyDBContext( keydb );
	cut = *pgpPeekContextMemPool( context );
	mgr = pgpPeekContextMemPoolMgr( context );

	sinfo = pgpSigToSigInfo( sig );
	strust = sinfo->trust;     
	strust &= ~PGP_SIGTRUSTF_CHECKED_TRIED;

	/* If the superior object is a userid, reset the WARNONLY flag. */
	parent = sig->up;
	if( OBJISUSERID(parent) )
	{
		uinfo = pgpUserIDToUserIDInfo( parent );
		if( uinfo->oldvalidity & PGP_USERIDTRUSTF_WARNONLY) 
		{
			uinfo->oldvalidity &= ~PGP_USERIDTRUSTF_WARNONLY;
			pgpKeyDBIsDirty( sigdb );
		}
	}

	/* The new hash style includes the userid header and count */
	sighashnew =  sinfo->version > PGPVERSION_3;

	/* Special verification rules for X.509 signatures */
	if( SIGISX509(sinfo) ) {
		strust |= PGP_SIGTRUSTF_TRIED;
		if( pgpSigX509Valid (sig, key) )
			strust |= PGP_SIGTRUSTF_CHECKED;
		goto done;
	}

	hash = pgpHashByNumber( (PGPHashAlgorithm)sinfo->hashalg );
	if( IsNull( hash ) )
		goto done;

	/* Try to create our hash context */
	hashctx = pgpHashCreate( mgr, hash );
	if( IsNull( hashctx ) )
	{
		err = kPGPError_OutOfMemory;
		goto done;
	}

	if( IsPGPError( err = pgpHashObjParents( hashctx, sig->up, sighashnew ) ) )
		goto done;

	buf = pgpFetchObject(sig, &len);
	extrabuf = pgpSigParseExtra(buf, len, &extralen);

	strust |= PGP_SIGTRUSTF_TRIED;

	/*
	 * extralen != 5 legal only if sig is on key directly,
	 * or it is a new-format EXTENDED type.
	 */
	if( !OBJISTOPKEY(sig->up)					/* Sig directly on key */
		&& extralen != 5						/* Traditional case */
		&& sinfo->version <= PGPVERSION_3)		/* New format packets */
	{
		/* Not legal, set as failed */
		goto done;
	}

	/* Hash in extra bytes for signature */
	PGPContinueHash(hashctx, extrabuf, extralen);

	if( sighashnew )
	{
		/* Make sure hash can't match any old or doc hashes */
		PGPByte postscript[6];
		postscript[0] = PGPVERSION_4;	/* actually a 4! */
		postscript[1] = 0xff;			/* different from sig type */
		postscript[2] = (PGPByte)(extralen>>24);
		postscript[3] = (PGPByte)(extralen>>16);
		postscript[4] = (PGPByte)(extralen>> 8);
		postscript[5] = (PGPByte)(extralen>> 0);
		PGPContinueHash (hashctx, postscript, sizeof(postscript));
	}

	/* Ready for the actual sig check */
	hlen = pgpPktBufferHeaderLen( buf );
	err = pgpSigDataCheckBuf(buf+hlen, len-hlen, key, pgpHashFinal(hashctx));
	PGPFreeHashContext( hashctx );
	hashctx = NULL;
	if( err != 1 && err != 0 )
	{
		/* ViaCrypt self-signs encryption-only keys (as they're RSA,
		   it's possible), but we don't handle these.  Ignore this
		   type of signature. */
		if( pgpIsKeyRelatedError( err ) )
		{
			err = kPGPError_NoErr;
		}
		goto done;
	}
	if( err == 1 )
	{
		/* Success */
		strust |= PGP_SIGTRUSTF_CHECKED;
		/*
		 * On finding a valid self signature, apply any info in sig
		 * to parent key which goes there.  "key" is signer, topkey,
		 * is top level key above this sig, parentkey is closest
		 * key above this sig (may be a subkey).
		 */
		/* Find the top level key and the key which is above us */
		if( OBJISTOPKEY( sig->up ) )
		{
			topkey = sig->up;
			parentkey = topkey;
		} else {
			topkey = sig->up->up;
			if( OBJISSUBKEY( sig->up ) )
				parentkey = sig->up;
			else
				parentkey = topkey;
		}
		pgpAssert( OBJISTOPKEY( topkey ) );
		pgpAssert( OBJISKEY( parentkey ) );

		if( key == topkey ) {
			/* Self signature */
			PGPByte const *pk;
			/* Look for expiration date, apply to key above us */
			/* Note that this may alter the contents of buf */
			pk = pgpSigFindSubpacket (buf, SIGSUB_KEY_EXPIRATION,
				0, NULL, NULL, NULL, NULL, NULL);
			if( IsntNull(pk) ) {
				PGPUInt32 keyexp;
				keyexp = (PGPUInt32)
					((unsigned)pk[0]<<8|pk[1]) << 16 |
					((unsigned)pk[2]<<8|pk[3]);
				kinfo = pgpKeyToKeyInfo( parentkey );
				kinfo->validityperiod =
						(PGPUInt16)(keyexp/(24*3600));
			}
			/* Also check for primary userid sigs here */
			if( OBJISUSERID( parent ) )
			{
				if( !(sinfo->trust & PGP_SIGTRUSTF_CHECKED) &&
					(strust & PGP_SIGTRUSTF_CHECKED) )
				{
					/* A freshly validated self-sig */
					if( SIGISPRIMARYUID (sinfo) )
					{
						/* Clear any manual primary uid settings we had */
						pgpKeyClearPrimaryUserIDs( key, (PGPUInt32)-1 );
					}
				}
			}
		}
	}
	err = kPGPError_NoErr;
 done:
	*verified = (strust & PGP_SIGTRUSTF_CHECKED) != 0;
	if( sinfo->trust != strust )
	{
		pgpKeyDBIsDirty( sigdb );
		sinfo->trust = strust;
		*changed = TRUE;
	}

	if( IsntNull( hashctx ) )
		PGPFreeHashContext( hashctx );
	if( IsntNull( mgr ) )
		pgpContextMemPoolCutBack( context, &cut );
	return err;
}



/*
 * Update the hash context with the data in the object.
 *
 * hc		HashContext to update
 * obj		Object to update it with
 * hashuseridlength	If true, include the length of a userid packet in hash
 *			(this is the post-V3 sig convention)
 *
 * Return	0 on success, or an error code
 */
	PGPError
pgpHashObj (PGPHashContext *hc, PGPKeyDBObj *obj, PGPBoolean hashuseridlength)
{
	PGPKeyDB *keydb;
	PGPUserIDInfo *uinfo;
	PGPSize objlen;
	PGPSize hlen;
	PGPByte const *objbuf;
	PGPByte tmpbuf[5];
	PGPBoolean mustfree = FALSE;

	if( IsNull(hc) || IsNull(obj) )
		return kPGPError_BadParams;

	if( OBJISKEY(obj) && pgpKeyIsSec(obj) )
	{
		objbuf = pgpKeyDBObjToPubData( obj, &objlen );
		mustfree = TRUE;
	} else {
		objbuf = pgpFetchObject(obj, &objlen);
	}
	if( IsNull(objbuf) )
	{
		keydb = PGPPeekKeyDBObjKeyDB( obj );
		return pgpKeyDBError(keydb);
	}

	hlen = pgpPktBufferHeaderLen( objbuf );
	objbuf += hlen;
	objlen -= hlen;

	/* We use this format even for subkeys */
	if( OBJISKEY(obj) )
	{
		pgpAssert(objlen <= 65535);
		tmpbuf[0] = PKTBYTE_BUILD(PKTBYTE_PUBKEY, 1);
		tmpbuf[1] = (PGPByte)(objlen>>8);
		tmpbuf[2] = (PGPByte)objlen;
		PGPContinueHash(hc, tmpbuf, 3);
	} else if( OBJISUSERID(obj) && hashuseridlength ) {
		/*
		 * Use four bytes for userid length for future expansion.  Can't do
		 * it for keys due to backwards compatibility.
		 */
		uinfo = pgpUserIDToUserIDInfo( obj );
		tmpbuf[0] = NAMEISATTR(uinfo)
							? PKTBYTE_BUILD(PKTBYTE_ATTRIBUTE, 0)
							: PKTBYTE_BUILD(PKTBYTE_USERID, 0);
		tmpbuf[1] = (PGPByte)(objlen>>24);
		tmpbuf[2] = (PGPByte)(objlen>>16);
		tmpbuf[3] = (PGPByte)(objlen>> 8);
		tmpbuf[4] = (PGPByte)(objlen>> 0);
		PGPContinueHash(hc, tmpbuf, 5);
	}
	PGPContinueHash(hc, objbuf, objlen);
	if( mustfree )
		PGPFreeData( (PGPByte *)objbuf-hlen );
	return kPGPError_NoErr;
}


/*
 * Update the hash context with the data from the object and its parents
 */
	PGPError
pgpHashObjParents (PGPHashContext *hc, PGPKeyDBObj *obj,
		 PGPBoolean hashuseridlength)
{
	PGPKeyDBObj *	parents[3];
	PGPError		err;
	int				level;

	/* Trace object's parents up to top level */
	level = 0;
	for ( ; ; ) {
		pgpAssert ((unsigned)level < sizeof(parents)/sizeof(parents[0]));
		parents[level++] = obj;
		if( OBJISTOPKEY(obj) )
			break;
		obj = obj->up;
	}

	/* Hash downwards from top to object */
	while (--level >= 0) {
		err = pgpHashObj( hc, parents[level], hashuseridlength );
		if( IsPGPError( err ) ) {
			return err;
		}
	}
	return kPGPError_NoErr;
}

/*
 * Sign the specified object obj, along with its parents.
 * Place signature into newly allocated sig buffer.
 * sig buffer should be at least pgpMakeSigMaxSize(spec) bytes lon
 * Returns size of sig in bytes on success, negative on error.
 */
	PGPInt32
pgpSignObj(PGPKeyDBObj *obj, PGPSigSpec *spec, PGPByte **sig)
{
	PGPContextRef context;
	PGPHashContext *hc;
	PGPInt32 len;
	PGPError retval;
	PGPBoolean sighashnew;
	MemPool	cut;
	PGPMemoryMgrRef mgr = NULL;

	/* Initialize hash */
	context = PGPPeekKeyDBContext(PGPPeekKeyDBObjKeyDB( obj ) );
	cut = *pgpPeekContextMemPool( context );
	mgr = pgpPeekContextMemPoolMgr( context );
	hc = pgpHashCreate( mgr, pgpSigSpecHash(spec));
	if( IsNull(hc) )
	{
		pgpContextMemPoolCutBack( context, &cut );
		return kPGPError_BadHashNumber;
	}

	/* Use new hashing convention (include userid length) on new formats */
	sighashnew = pgpSigSpecVersion(spec) > PGPVERSION_3;

	retval = pgpHashObjParents( hc, obj, sighashnew );
	if( IsPGPError( retval ) )
	{
		PGPFreeHashContext(hc);
		pgpContextMemPoolCutBack( context, &cut );
		return (PGPInt32) retval;
	}

	/* Make the new sig */
	len = pgpMakeSig (spec, hc, sig);
	PGPFreeHashContext(hc);
	pgpContextMemPoolCutBack( context, &cut );
	return len;
}



/*
 * Local Variables:
 * tab-width: 4
 * End:
 * vi: ts=4 sw=4
 * vim: si
 */
